-- Demoflex Ultimate Reflex UI v2.0-release3
-- (version should always match the Demoflex version it was written for)

-- This Lua sourcecode © 2015 Jonathan Richman AKA Qualx or Qualx Halfbeard, all
-- rights reserved.

-- Demoflex Ultimate © 2015 Benjamin Schmitt, and Obscure Systems GmbH.

-- This code was written for use with Demoflex Ultimate, and may not be modified
-- (except for bug fixes) or reproduced in any form, nor may any part of it be
-- used for any purpose other than that for which it was intended, without
-- written permission from the author. You are allowed to display the HUD
-- defined by this code whenever you would be allowed to display the Demoflex
-- interface.

-- You are responsible for any code that you run on your computer. I do not
-- accept responsibility for any of installing or running this code, including,
-- but not limited to, loss of data.

-- The entry box cursor positioning code is copied and modified from stock
-- Reflex code (specifically from uiEditBox in reflexcore.lua)

-- This code was written for and tested with Reflex Alpha 0.34.2.

Demoflex = {canPosition = false; canHide = false; isMenu = true;}
registerWidget("Demoflex")

Demoflex_ActionSafe = {canPosition = false; canHide = false; isMenu = true;}
registerWidget("Demoflex_ActionSafe")

Demoflex_Sidebar = {canPosition = false; canHide = false; isMenu = true;}
registerWidget("Demoflex_Sidebar")

Demoflex_Cmd = {canPosition = false; canHide = false; isMenu = true;}
registerWidget("Demoflex_Cmd")

Demoflex_Cmd.wrapCommands =
{
	"ui_demoflex_hide_toggle",	-- lol massive hax
	"re_add_keyframe",
	"re_edit_toggle",
	"re_next_keyframe",
	"re_prev_keyframe",
	"re_remove_keyframe",
}

local black =
{
	r=0x00;
	g=0x00;
	b=0x00;
	a=0xff;
}

local veryDarkGrey =
{
	r=0x1a;
	g=0x1a;
	b=0x1a;
	a=0xff;
}

local darkGrey =
{
	r=0x28;
	g=0x28;
	b=0x28;
	a=0xff;
}

local lessDarkGrey =
{
	r=0x3a;
	g=0x3a;
	b=0x3a;
	a=0xff;
}

local midGrey =
{
	r=0x4b;
	g=0x4b;
	b=0x4b;
	a=0xff;
}

local lightGrey =
{
	r=0x80;
	g=0x80;
	b=0x80;
	a=0xff;
}

local lightGreyHighAlpha =
{
	r=0x80;
	g=0x80;
	b=0x80;
	a=0xd0;
}

local lightGreyMidAlpha =
{
	r=0x80;
	g=0x80;
	b=0x80;
	a=0xa0;
}

local lightGreyLowAlpha =
{
	r=0x80;
	g=0x80;
	b=0x80;
	a=0x50;
}

local veryLightGrey =
{
	r=0xd0;
	g=0xd0;
	b=0xd0;
	a=0xff;
}

local white =
{
	r=0xff;
	g=0xff;
	b=0xff;
	a=0xff;
}

local blue =
{
	r=0x00;
	g=0x5A;
	b=0x96;
	a=0xff;
}

local lightBlue =
{
	r=0x20;
	g=0x7A;
	b=0xb6;
	a=0xff;
}

local fontFaceRegular = "SourceSansPro-Regular"
local fontFaceBold = "SourceSansPro-Semibold"
local fontFaceLogo = "InversionzUnboxed"

local sidebarWidth
local sizeS
local sizeM
local sizeL
local sizeDivider
local topMargin
local horizontalMargin
local bottomMargin
local fontSizeTab
local fontSizeS
local fontSizeM
local fontSizeL
local controlSizeS
local controlSizeM
local controlSizeL

local function rescaleHud(scale)
	-- bug: does not recalculate if ui_viewport_height is changed manually
	if not scale then scale = 0 end
	local resolution = (consoleGetVariable("r_fullscreen") >= 1) and consoleGetVariable("r_resolution_fullscreen") or consoleGetVariable("r_resolution_windowed")
	local height = resolution and resolution[2] or 0
	local viewportHeight = math.max(640,math.floor(0.5+height/(scale > 0 and scale or 1)))
	if consoleGetVariable("ui_viewport_height") ~= viewportHeight or scale == 0 then
		sidebarWidth = math.ceil(viewportHeight*2/5)

		topMargin = math.ceil(10*viewportHeight/640)
		horizontalMargin = math.ceil(20*viewportHeight/640)
		bottomMargin = math.ceil(10*viewportHeight/640)

		sizeS = math.ceil(30*viewportHeight/640)
		sizeM = math.ceil(33*viewportHeight/640)
		sizeL = math.ceil(36*viewportHeight/640)

		sizeDivider = math.ceil(10*viewportHeight/640)

		fontSizeTab = math.ceil(12*viewportHeight/640)
		fontSizeS = math.ceil(18*viewportHeight/640)
		fontSizeM = math.ceil(19.8*viewportHeight/640)
		fontSizeL = math.ceil(21.6*viewportHeight/640)

		controlSizeS = math.ceil(24*viewportHeight/640)
		controlSizeM = math.ceil(26.4*viewportHeight/640)
		controlSizeL = math.ceil(28.8*viewportHeight/640)

		if scale > 0 then consolePerformCommand(string.format("ui_viewport_height %i",viewportHeight)) end
	end
end

rescaleHud(0)

local initializedAll = false
local initializedDemoflex = false
local initializedDemoflexCmd = false
local initializedDemoflexSidebar = false
local initializedDemoflexActionSafe = false

local enabled = false
local re_edit_toggle = false
local hidden = false
local lastReplayName
local replayNameBeforeRename
local replayNameAfterRename

local function wrapBoundCommands()
	for _,command in pairs(Demoflex_Cmd.wrapCommands) do
		local keybind = bindReverseLookup(command)
		while keybind ~= "(unbound)" do
			consolePerformCommand(string.format("bind %s ui_demoflex_cmd_exec %s",keybind,command))
			keybind = bindReverseLookup(command)
		end
	end
end

local function loadDemoflex()
	enabled = true
	consolePerformCommand("ui_demoflex_cmd_exec (none)")
	local doBackup = consoleGetVariable("ui_demoflex_backup_config") or 0
	if doBackup >= 1 then
		consolePerformCommand("ui_demoflex_enable 0")
		consolePerformCommand("saveconfig demoflex_hud_game_backup")
		consolePerformCommand("ui_demoflex_enable 1")
	end
	wrapBoundCommands()
end

local function unloadDemoflex()
	enabled = false
	local doBackup = consoleGetVariable("ui_demoflex_backup_config") or 0
	if doBackup >= 1 then
		-- this introduces a bug - if backup_config is set AFTER loading demoflex, problems can occur. Probably not important.
		consolePerformCommand("loadconfig demoflex_hud_game_backup")
		consolePerformCommand("ui_demoflex_cmd_exec (none)")
		consolePerformCommand("ui_demoflex_enable 0")
	end
end

local function scaleTransition(x) return (x*(2-x))^2 end

local function button(x,y,w,h,id,fontSize,label,enabled,highlighted,locked)
	local border = 4
	local cornerRadius = 3.0
	if not fontSize then fontSize = math.ceil((h-border)*0.75) end
	local mouseState
	if enabled and not locked then
		mouseState = mouseRegion(x, y, w, h, id);
	else
		mouseState = {}
	end
	nvgSave()
	nvgBeginPath()
	nvgRoundedRect(x+border/2,y+border/2,w-border,h-border,cornerRadius)
	nvgStrokeWidth(border)
	nvgStrokeColor(midGrey)
	nvgStroke()
	if not enabled then
		nvgFillColor(lessDarkGrey)
	elseif highlighted then
		nvgFillColor(mouseState.hover and lightBlue or blue)
	else
		nvgFillColor(mouseState.hover and lightBlue or veryDarkGrey)
	end
	nvgFill()
	nvgFontFace(fontFaceRegular)
	nvgFontSize(fontSize)
	nvgFontBlur(0)
	nvgTextAlign(NVG_ALIGN_LEFT,NVG_ALIGN_TOP)
	local textWidth = nvgTextWidth(label)
	local textX = math.floor(0.5+x+(w-textWidth)/2)
	local textY = math.floor(0.5+y+(h-fontSize)/2)
	nvgFillColor(enabled and white or lightGrey)
	nvgText(textX,textY,label)
	nvgRestore()
	return mouseState.leftUp, mouseState.hover
end

local function getConsoleCommandFunc(command)
	return function() consolePerformCommand(command) end
end

local function dialogBox(x,y,text,entryWidth,buttonLabels,callback)
	local titleBarHeight = 24
	local horizontalMargin = 10
	local verticalSpacing = 8
	local verticalMargin = 8
	local fontSize = 20
	local buttonWidth = 120
	local buttonHeight = 30
	local buttonSpacing = 16
	local buttonFontSize = 24
	local buttonBorder = 1
	local buttonCornerRadius = 3.0
	local entryFullHeight = 30
	local entrySpacing = 4
	local entryFontSize = 22
	local entryBorder = 1
	local entryFullWidth = entryWidth + entryBorder*2
	local entryHeight = entryFullHeight - entryBorder*2
	local entryHorizontalPadding = 4
	local entryVerticalPadding = math.floor(0.5+(entryHeight - entryFontSize)/2)
	local border = 1
	local cursorFlash = 0
	local cursorFlashTime = 0.5
	local cursorFlashState = true

	if not text then return end
	local buttonCount = #buttonLabels

	if buttonCount == 0 then
		buttonSpacing = 0
		buttonWidth = 0
		buttonHeight = -verticalSpacing
		verticalMargin = 2*verticalMargin
	end

	nvgSave()
	local buttonsWidth = buttonCount*buttonWidth + (buttonCount - 1)*buttonSpacing
	local textWidth = buttonsWidth
	local textHeight
	local drawText
	nvgFontSize(fontSize)
	nvgFontFace(fontFaceRegular)
	nvgFontBlur(0)
	nvgTextAlign(NVG_ALIGN_LEFT,NVG_ALIGN_TOP)

	local lineTextWidth = nvgTextWidth(text)
	local lineWidth = lineTextWidth + entrySpacing + entryWidth + 2*entryBorder
	textWidth = math.max(textWidth,lineWidth)
	textHeight = math.max(fontSize,entryFullHeight)
	local indentTextX = math.floor(0.5+(textWidth - lineWidth)/2)
	local indentTextY = math.floor(0.5+(textHeight - fontSize)/2)
	local indentEntryX = indentTextX+lineTextWidth+entrySpacing
	local indentEntryY = math.floor(0.5+(textHeight - entryFullHeight)/2)
	local indentEntryTextX = indentEntryX + entryHorizontalPadding
	local indentEntryTextY = indentEntryY + entryVerticalPadding
	drawText = function(x,y,initialEntryString)
		nvgTextAlign(NVG_ALIGN_LEFT,NVG_ALIGN_TOP)
		nvgFontSize(fontSize)
		nvgFontFace(fontFaceRegular)
		nvgFontBlur(0)
		nvgFillColor(white)
		nvgText(x+indentTextX,y+indentTextY,text)

		local dialogEntryFull = mouseRegion(x+indentEntryX,y+indentEntryY,entryWidth,entryHeight)
		local dialogEntry = textRegion(x+indentEntryTextX,y+indentEntryTextY,entryWidth-entryHorizontalPadding*2,entryHeight-entryVerticalPadding/2,initialEntryString or "")

		nvgBeginPath()
		nvgRect(x+indentEntryX+entryBorder/2,y+indentEntryY+entryBorder/2,entryWidth+entryBorder,entryHeight+entryBorder)
		nvgStrokeWidth(entryBorder)
		nvgStrokeColor((dialogEntry.focus or dialogEntryFull.hover) and lightBlue or blue)
		nvgStroke()

		local entryText = dialogEntry.text

		nvgFontSize(entryFontSize)

		if dialogEntry.focus then
			if dialogEntry.cursorChanged then
				cursorFlash = 0
				cursorFlashState = true
			else
				cursorFlash = cursorFlash + deltaTimeRaw
				if cursorFlash >= cursorFlashTime then
					cursorFlashState = not cursorFlashState
					cursorFlash = cursorFlash - cursorFlashTime
				end
			end

			local cursor = dialogEntry.cursor
			local cursorStart = dialogEntry.cursorStart

			if cursor ~= cursorStart then
				local selectionBegin = math.min(cursor,cursorStart)
				local selectionEnd = math.max(cursor,cursorStart)

				nvgBeginPath()
				nvgRect(x+indentEntryTextX+nvgTextWidth(string.sub(entryText,1,selectionBegin)),y+indentEntryTextY,nvgTextWidth(string.sub(entryText,selectionBegin+1,selectionEnd)),entryFontSize)
				nvgFillColor(midGrey)
				nvgFill()
			end

			if cursorFlashState then
				nvgBeginPath()
				nvgStrokeWidth(1)
				nvgStrokeColor(lightGrey)
				local cursorPosition = 0.5+x+indentEntryTextX+((cursor > 0) and nvgTextWidth(string.sub(entryText,1,cursor)) or 0)
				nvgMoveTo(cursorPosition,y+indentEntryTextY)
				nvgLineTo(cursorPosition,y+indentEntryTextY+entryFontSize)
				nvgStroke()
			end

			-- this bit stolen from uiEditBox(), thanks TurboPixel!
			if (dialogEntry.leftDown or dialogEntry.leftHeld) and dialogEntry.mouseInside then
				local mousex = dialogEntry.mousex;
				local lentext = string.len(dialogEntry.text);
				local prevDistanceFromCursor;
				local newCusror = lentext;
				for l = 0, lentext do
					local s = string.sub(dialogEntry.text, 0, l);
					local tw = nvgTextWidth(s);
					local endtext = x+indentEntryTextX + tw;

					local distanceFromCursor = math.abs(endtext - dialogEntry.mousex);

					if l > 0 then
						if distanceFromCursor > prevDistanceFromCursor then
							newCusror = l-1;
							break;
						end
					end

					prevDistanceFromCursor = distanceFromCursor;
				end
				local dragSelection = dialogEntry.leftHeld and not dialogEntry.leftDown;
				dialogEntry.cursorStart, dialogEntry.cursor = textRegionSetCursor(dialogEntry.id, newCusror, dragSelection);
			end
		else
			cursorFlash = 0
			cursorFlashState = true
		end

		nvgFillColor(white)
		nvgText(x+indentEntryTextX,y+indentEntryTextY,entryText..".rep")

		return dialogEntry
	end

	nvgRestore()

	local innerWidth = textWidth+2*horizontalMargin
	local innerHeight = textHeight+buttonHeight+2*verticalMargin+verticalSpacing
	local outerWidth = innerWidth+2*border
	local outerHeight = innerHeight+2*border+titleBarHeight

	if not x then x = -math.floor(0.5+outerWidth/2) end
	if not y then y = -math.floor(0.5+outerHeight/2) end

	local textX = (innerWidth-textWidth)/2

	local titleBarClick = nil

	return function(initialEntryString)
		local id = 1
		nvgSave()
		if viewport.height % 2 == 1 then nvgTranslate(0,-0.5) end
		if viewport.width % 2 == 1 then nvgTranslate(-0.5,0) end
		nvgTranslate(x,y)

		local titleBarMouseRegion = mouseRegion(0, 0, outerWidth, titleBarHeight, id)
		id = id + 1

		local mousePos = {titleBarMouseRegion.mousex,titleBarMouseRegion.mousey}

		if titleBarMouseRegion.leftDown then
			titleBarClick = mousePos
		elseif titleBarClick and titleBarMouseRegion.leftHeld then
			nvgTranslate(mousePos[1]-titleBarClick[1],mousePos[2]-titleBarClick[2])
		elseif titleBarClick and titleBarMouseRegion.leftUp then
			nvgTranslate(mousePos[1]-titleBarClick[1],mousePos[2]-titleBarClick[2])
			x = x+(mousePos[1]-titleBarClick[1])
			y = y+(mousePos[2]-titleBarClick[2])
		end

		nvgBeginPath()
		nvgRect(0,0,outerWidth,outerHeight)
		nvgFillColor(blue)
		nvgFill()
		nvgTranslate(border,border+titleBarHeight)
		nvgBeginPath()
		nvgRect(0,0,innerWidth,innerHeight)
		nvgFillColor(black)
		nvgFill()
		local dialogEntry = drawText(horizontalMargin,verticalMargin,initialEntryString)

		for i=1,buttonCount do
			local label = buttonLabels[i]
			local buttonX = math.floor(0.5+(innerWidth - buttonsWidth)/2) + (i-1)*(buttonWidth+buttonSpacing)
			local buttonY = verticalMargin+textHeight+verticalSpacing
			local mouseState = mouseRegion(buttonX, buttonY, buttonWidth, buttonHeight, id)
			nvgBeginPath()
			nvgRoundedRect(buttonX+buttonBorder/2,buttonY+buttonBorder/2,buttonWidth-buttonBorder,buttonHeight-buttonBorder,buttonCornerRadius)
			nvgStrokeWidth(buttonBorder)
			nvgStrokeColor(blue)
			nvgFillColor(mouseState.hover and lightBlue or black)
			nvgFill()
			nvgStroke()
			nvgFontFace(fontFaceRegular)
			nvgFontSize(buttonFontSize)
			nvgFontBlur(0)
			nvgTextAlign(NVG_ALIGN_LEFT,NVG_ALIGN_TOP)
			local buttonTextWidth = nvgTextWidth(label)
			nvgFillColor(white)
			nvgText(buttonX+math.floor(0.5+(buttonWidth-buttonTextWidth)/2),math.floor(0.5+buttonY+(buttonHeight-buttonFontSize)/2),label)

			if mouseState.leftUp then callback(i,dialogEntry.text) end
			id = id + 1
		end

		nvgRestore()
		return dialogEntry.text
	end
end

local function messageBox(x,y,text,buttonLabels,callback)	-- x = nil or y = nil will center the box in the relevant dimension, buttons should be {label,callback} pairs
	local titleBarHeight = 24
	local horizontalMargin = 10
	local verticalSpacing = 8
	local verticalMargin = 8
	local fontSize = 20
	local buttonWidth = 120
	local buttonHeight = 30
	local buttonSpacing = 16
	local buttonFontSize = 24
	local border = 1
	if not text then return end
	local buttonCount = buttonLabels and #buttonLabels or 0

	if buttonCount == 0 then
		buttonSpacing = 0
		buttonWidth = 0
		buttonHeight = -verticalSpacing
		verticalMargin = 2*verticalMargin
	end

	nvgSave()

	local buttonsWidth = buttonCount*buttonWidth + (buttonCount - 1)*buttonSpacing
	local textWidth = buttonsWidth
	local textHeight
	local drawText
	nvgFontSize(fontSize)
	nvgFontFace(fontFaceRegular)
	nvgFontBlur(0)
	nvgTextAlign(NVG_ALIGN_LEFT,NVG_ALIGN_TOP)
	
	local lineWidth = nvgTextWidth(text)
	textWidth = math.max(textWidth,lineWidth)
	local indent = math.floor(0.5+(textWidth - lineWidth)/2)
	drawText = function(x,y)
		nvgTextAlign(NVG_ALIGN_LEFT,NVG_ALIGN_TOP)
		nvgFontSize(fontSize)
		nvgFontFace(fontFaceRegular)
		nvgFontBlur(0)
		nvgFillColor(white)
		nvgText(x+indent,y,text)
	end
	textHeight = fontSize

	nvgRestore()

	local innerWidth = textWidth+2*horizontalMargin
	local innerHeight = fontSize+buttonHeight+2*verticalMargin+verticalSpacing
	local outerWidth = innerWidth+2*border
	local outerHeight = innerHeight+2*border+titleBarHeight

	if not x then x = -math.floor(0.5+outerWidth/2) end
	if not y then y = -math.floor(0.5+outerHeight/2) end

	local textX = (innerWidth-textWidth)/2

	local titleBarClick = nil

	return function()
		local id = 1
		nvgSave()
		if viewport.height % 2 == 1 then nvgTranslate(0,-0.5) end
		if viewport.width % 2 == 1 then nvgTranslate(-0.5,0) end
		nvgTranslate(x,y)

		local titleBarMouseRegion = mouseRegion(0, 0, outerWidth, titleBarHeight, id)
		id = id + 1

		local mousePos = {titleBarMouseRegion.mousex,titleBarMouseRegion.mousey}

		if titleBarMouseRegion.leftDown then
			titleBarClick = mousePos
		elseif titleBarClick and titleBarMouseRegion.leftHeld then
			nvgTranslate(mousePos[1]-titleBarClick[1],mousePos[2]-titleBarClick[2])
		elseif titleBarClick and titleBarMouseRegion.leftUp then
			nvgTranslate(mousePos[1]-titleBarClick[1],mousePos[2]-titleBarClick[2])
			x = x+(mousePos[1]-titleBarClick[1])
			y = y+(mousePos[2]-titleBarClick[2])
		end

		nvgBeginPath()
		nvgRect(0,0,outerWidth,outerHeight)
		nvgFillColor(blue)
		nvgFill()
		nvgTranslate(border,border+titleBarHeight)
		nvgBeginPath()
		nvgRect(0,0,innerWidth,innerHeight)
		nvgFillColor(black)
		nvgFill()
		drawText((innerWidth-textWidth)/2,verticalMargin)
		for i=1,buttonCount do
			local label = buttonLabels[i]
			local buttonX = math.floor(0.5+(innerWidth - buttonsWidth)/2) + (i-1)*(buttonWidth+buttonSpacing)
			local buttonY = verticalMargin+textHeight+verticalSpacing
			local buttonBorder = 1
			local cornerRadius = 3.0
			local mouseState = mouseRegion(buttonX, buttonY, buttonWidth, buttonHeight, id)
			nvgBeginPath()
			nvgRoundedRect(buttonX+buttonBorder/2,buttonY+buttonBorder/2,buttonWidth-buttonBorder,buttonHeight-buttonBorder,cornerRadius)
			nvgStrokeWidth(buttonBorder)
			nvgStrokeColor(blue)
			nvgFillColor(mouseState.hover and lightBlue or black)
			nvgFill()
			nvgStroke()
			nvgFontFace(fontFaceRegular)
			nvgFontSize(buttonFontSize)
			nvgFontBlur(0)
			nvgTextAlign(NVG_ALIGN_LEFT,NVG_ALIGN_TOP)
			local buttonTextWidth = nvgTextWidth(label)
			nvgFillColor(white)
			nvgText(buttonX+math.floor(0.5+(buttonWidth-buttonTextWidth)/2),math.floor(0.5+buttonY+(buttonHeight-buttonFontSize)/2),label)

			if mouseState.leftUp then callback(i) end
			id = id + 1
		end

		nvgRestore()
	end
end

local function timedMessageBox(x,y,text,buttonLabels,callback,timeLimit)	-- x = nil or y = nil will center the box in the relevant dimension, buttons should be {label,callback} pairs
	local titleBarHeight = 24
	local horizontalMargin = 10
	local verticalSpacing = 8
	local verticalMargin = 8
	local fontSize = 20
	local buttonWidth = 120
	local buttonHeight = 30
	local buttonSpacing = 16
	local buttonFontSize = 24
	local border = 1
	if not text then return end

	local timer = timeLimit

	local buttonCount = buttonLabels and #buttonLabels or 0

	if buttonCount == 0 then
		buttonSpacing = 0
		buttonWidth = 0
		buttonHeight = -verticalSpacing
		verticalMargin = 2*verticalMargin
	end

	nvgSave()

	local buttonsWidth = buttonCount*buttonWidth + (buttonCount - 1)*buttonSpacing
	local textWidth = buttonsWidth
	local textHeight
	local drawText
	nvgFontSize(fontSize)
	nvgFontFace(fontFaceRegular)
	nvgFontBlur(0)
	nvgTextAlign(NVG_ALIGN_LEFT,NVG_ALIGN_TOP)
	
	local lineWidth = nvgTextWidth(text)
	textWidth = math.max(textWidth,lineWidth)
	local indent = math.floor(0.5+(textWidth - lineWidth)/2)
	drawText = function(x,y)
		nvgTextAlign(NVG_ALIGN_LEFT,NVG_ALIGN_TOP)
		nvgFontSize(fontSize)
		nvgFontFace(fontFaceRegular)
		nvgFontBlur(0)
		nvgFillColor(white)
		nvgText(x+indent,y,string.format(text,math.ceil(math.max(0,timer))))
	end
	textHeight = fontSize

	nvgRestore()

	local innerWidth = textWidth+2*horizontalMargin
	local innerHeight = fontSize+buttonHeight+2*verticalMargin+verticalSpacing
	local outerWidth = innerWidth+2*border
	local outerHeight = innerHeight+2*border+titleBarHeight

	if not x then x = -math.floor(0.5+outerWidth/2) end
	if not y then y = -math.floor(0.5+outerHeight/2) end

	local textX = (innerWidth-textWidth)/2

	local titleBarClick = nil
	return function(reset,...)
		local id = 1
		if reset then timer = timeLimit else timer = timer - deltaTimeRaw end

		nvgSave()
		if viewport.height % 2 == 1 then nvgTranslate(0,-0.5) end
		if viewport.width % 2 == 1 then nvgTranslate(-0.5,0) end
		nvgTranslate(x,y)

		local titleBarMouseRegion = mouseRegion(0, 0, outerWidth, titleBarHeight, id)
		id = id + 1

		local mousePos = {titleBarMouseRegion.mousex,titleBarMouseRegion.mousey}

		if titleBarMouseRegion.leftDown then
			titleBarClick = mousePos
		elseif titleBarClick and titleBarMouseRegion.leftHeld then
			nvgTranslate(mousePos[1]-titleBarClick[1],mousePos[2]-titleBarClick[2])
		elseif titleBarClick and titleBarMouseRegion.leftUp then
			nvgTranslate(mousePos[1]-titleBarClick[1],mousePos[2]-titleBarClick[2])
			x = x+(mousePos[1]-titleBarClick[1])
			y = y+(mousePos[2]-titleBarClick[2])
		end

		nvgBeginPath()
		nvgRect(0,0,outerWidth,outerHeight)
		nvgFillColor(blue)
		nvgFill()
		nvgTranslate(border,border+titleBarHeight)
		nvgBeginPath()
		nvgRect(0,0,innerWidth,innerHeight)
		nvgFillColor(black)
		nvgFill()
		drawText((innerWidth-textWidth)/2,verticalMargin)
		for i=1,buttonCount do
			local label = buttonLabels[i]
			local buttonX = math.floor(0.5+(innerWidth - buttonsWidth)/2) + (i-1)*(buttonWidth+buttonSpacing)
			local buttonY = verticalMargin+textHeight+verticalSpacing
			local buttonBorder = 1
			local cornerRadius = 3.0
			local mouseState = mouseRegion(buttonX, buttonY, buttonWidth, buttonHeight, id)
			nvgBeginPath()
			nvgRoundedRect(buttonX+buttonBorder/2,buttonY+buttonBorder/2,buttonWidth-buttonBorder,buttonHeight-buttonBorder,cornerRadius)
			nvgStrokeWidth(buttonBorder)
			nvgStrokeColor(blue)
			nvgFillColor(mouseState.hover and lightBlue or black)
			nvgFill()
			nvgStroke()
			nvgFontFace(fontFaceRegular)
			nvgFontSize(buttonFontSize)
			nvgFontBlur(0)
			nvgTextAlign(NVG_ALIGN_LEFT,NVG_ALIGN_TOP)
			local buttonTextWidth = nvgTextWidth(label)
			nvgFillColor(white)
			nvgText(buttonX+math.floor(0.5+(buttonWidth-buttonTextWidth)/2),math.floor(0.5+buttonY+(buttonHeight-buttonFontSize)/2),label)

			if timer > 0 and mouseState.leftUp then callback(i) end
			id = id + 1
		end

		nvgRestore()
		if timer <= 0 then callback(0) return true else return false end
	end
end

function Demoflex:initialize()
	widgetCreateConsoleVariable("enable","int",0)
	widgetCreateConsoleVariable("backup_config","int",1)
	initializedDemoflex = true
end

Demoflex_Cmd.__mbStack = {}
Demoflex_Cmd.__mbStackArgs = {}
Demoflex_Cmd.__mbStackTop = 0

function Demoflex_Cmd:pushMessageBox(mbox,...)
	local mbStack = self.__mbStack
	local mbStackArgs = self.__mbStackArgs
	local mbStackTop = self.__mbStackTop
	mbStackTop = mbStackTop + 1
	mbStack[mbStackTop] = mbox
	local args = {...}
	mbStackArgs[mbStackTop] = args
	self.drawMessageBox = mbox
	self.__mbStackTop = mbStackTop
end

function Demoflex_Cmd:popMessageBox()
	local mbStack = self.__mbStack
	local mbStackArgs = self.__mbStackArgs
	local mbStackTop = self.__mbStackTop
	mbStack[mbStackTop] = nil
	mbStackArgs[mbStackTop] = nil
	mbStackTop = mbStackTop - 1
	if mbStackTop > 0 then self.drawMessageBox = mbStack[mbStackTop] else self.drawMessageBox = nil end
	self.__mbStackTop = mbStackTop
end

function Demoflex_Cmd:initialize()
	widgetCreateConsoleVariable("exec","string","(none)")
	self.mbJump = messageBox(nil,nil,"Jumping, please wait...")
	initializedDemoflexCmd = true
end

function Demoflex_ActionSafe:initialize()
	widgetCreateConsoleVariable("show","int","1")
	initializedDemoflexActionSafe = true
end

local function doSave(buttonCode,name)
	if name then name = string.gsub(name," ","_") else name = "" end
	if buttonCode == 0 or buttonCode == 1 or buttonCode == 2 then
		if name ~= "" then
			consolePerformCommand(string.format("re_save %s",name))
			Demoflex_Cmd:popMessageBox()
			Demoflex_Sidebar.showSave = false
			replayNameBeforeRename = replayName
			replayNameAfterRename = name
			if buttonCode ~= 2 then Demoflex_Cmd:pushMessageBox(Demoflex_Sidebar.mbSaveDone)
			else Demoflex_Cmd:pushMessageBox(Demoflex_Sidebar.mbSaveDoneNowQuit,true) end
		else
			Demoflex_Cmd:pushMessageBox(Demoflex_Sidebar.mbSaveErrorEmptyName)
		end
	elseif buttonCode == 3 then
		Demoflex_Cmd:popMessageBox()
		Demoflex_Sidebar.showSave = false
	end
end

local function doSaveDone(buttonCode)
	Demoflex_Cmd:popMessageBox()
end

local function doSaveDoneNowQuit(buttonCode)
	Demoflex_Cmd:popMessageBox()
	consolePerformCommand("ui_demoflex_cmd_exec quit")
end

local function doSaveErrorSeen(buttonCode)
	Demoflex_Cmd:popMessageBox()
end

local function doReset(buttonCode)
	Demoflex_Cmd:popMessageBox()
	Demoflex_Sidebar.showReset = false
	if buttonCode == 0 or buttonCode == 1 then consolePerformCommand(string.format("play %s",replayName)) consolePerformCommand("re_edit_toggle") end
end

function Demoflex_Sidebar:initialize()
	self.mbSave = dialogBox(nil,nil,"Replay name:",500,{"Save","Save & Quit","Cancel"},doSave)
	self.mbSaveDone = messageBox(nil,nil,"Saved!",{"OK"},doSaveDone)
	self.mbSaveDoneNowQuit = timedMessageBox(nil,nil,"Saved! Qutting in %i seconds...",{"Quit Now"},doSaveDoneNowQuit,3)
	self.mbSaveErrorEmptyName = messageBox(nil,nil,"Replay name cannot be empty!",{"OK"},doSaveErrorSeen)
	self.mbReset = messageBox(nil,nil,"You will lose any unsaved changes! Are you sure you want to reload this replay?",{"Yes","No"},doReset)
	initializedDemoflexSidebar = true
end

function Demoflex:draw()
	local isEnabled = (widgetGetConsoleVariable("enable") == 1)
	if isEnabled then rescaleHud(1) end
	if not initializedAll then
		if initializedDemoflex and initializedDemoflexCmd and initializedDemoflexSidebar and initializedDemoflexActionSafe then
			if isEnabled then
				enabled = true
				re_edit_toggle = false
				lastReplayName = replayName
				wrapBoundCommands()
			else
				enabled = false
				re_edit_toggle = false
			end
			initializedAll = true
		end
		return
	end
	if not enabled and isEnabled then
		loadDemoflex()
	elseif enabled and not isEnabled then
		unloadDemoflex()
	end
end

function Demoflex_Cmd:draw()
	if not initializedAll or not enabled then return end
	if replayName == "" or replayName == "menu" then return end
	if lastReplayName ~= replayName then
		widgetSetConsoleVariable("exec","(none)")
		re_edit_toggle = false
		lastReplayName = replayName
	elseif self.nextJump then
		local jump = self.nextJump
		self.nextJump = nil
		consolePerformCommand(jump)
		self:popMessageBox()
	else
		local cmd = widgetGetConsoleVariable("exec")
		if cmd and cmd ~= "" and cmd ~= "(none)" then
			widgetSetConsoleVariable("exec","(none)")
			if cmd == "ui_demoflex_hide_toggle" then
				hidden = not hidden
				consolePerformCommand(string.format("cl_show_hud %i",hidden and 0 or 1))
			elseif cmd == "re_edit_toggle" then
				re_edit_toggle = not re_edit_toggle
				consolePerformCommand(cmd)
			elseif cmd == "re_next_keyframe" or cmd == "re_prev_keyframe" or cmd == "re_next_marker" or cmd == "re_prev_marker" then
				self:pushMessageBox(self.mbJump)
				self.nextJump = cmd
			else
				consolePerformCommand(cmd)
			end
		end
	end
	if self.drawMessageBox then self.__mbStackArgs[self.__mbStackTop] = {self.drawMessageBox(unpack(self.__mbStackArgs[self.__mbStackTop]))} end
end

function Demoflex_ActionSafe:draw()
	if not initializedAll or not enabled or hidden or replayName == "" or replayName == "menu" then return end
	local x = -viewport.width/2
	local y = -viewport.height/2
	local w = viewport.width
	local h = viewport.height

	nvgBeginPath()
	nvgSave()
	nvgScale(0.97,0.97)
	nvgRect(x,y,w,h)
	nvgRestore()
	nvgSave()
	nvgScale(0.85,0.85)
	nvgRect(x,y,w,h)
	nvgRestore()
	nvgStrokeWidth(3)
	nvgStrokeColor(lightGreyMidAlpha)
	nvgStroke()

	nvgBeginPath()
	nvgCircle(0,0,0.05*viewport.height/2)
	nvgStrokeWidth(3)
	nvgStrokeColor(lightGreyMidAlpha)
	nvgStroke()

	nvgBeginPath()
	nvgCircle(0,0,0.2*viewport.height/2)
	nvgStrokeWidth(3)
	nvgStrokeColor(lightGreyLowAlpha)
	nvgStroke()

	nvgBeginPath()
	nvgMoveTo(0,0.2*viewport.height/2+1)
	nvgLineTo(0,0.5*viewport.height/2)
	nvgMoveTo(0.2*viewport.height/2+1,0)
	nvgLineTo(0.5*viewport.width/2,0)
	nvgMoveTo(0,-0.2*viewport.height/2-1)
	nvgLineTo(0,-0.5*viewport.height/2)
	nvgMoveTo(-0.2*viewport.height/2-1,0)
	nvgLineTo(-0.5*viewport.width/2,0)
	nvgStrokeWidth(2)
	nvgStrokeColor(lightGreyLowAlpha)
	nvgStroke()
end

local sidebarControls =
{
	{"label","Keyframe Controls"},
	{"button","Add Keyframe",getConsoleCommandFunc("ui_demoflex_cmd_exec re_add_keyframe")},
	{"button","Remove Keyframe",getConsoleCommandFunc("ui_demoflex_cmd_exec re_remove_keyframe")},
	{"button","Jump to next keyframe",getConsoleCommandFunc("ui_demoflex_cmd_exec re_next_keyframe")},
	{"button","Jump to prev keyframe",getConsoleCommandFunc("ui_demoflex_cmd_exec re_prev_keyframe")},
	{"divider"},
	{"label","Camera Controls"},
	{"sublabel","Angle Interpolation"},
	{"multibutton",{"none","linear","spline"},function(interp) if interp == "spline" then consolePerformCommand("re_set_angle_lerp quaternion") else consolePerformCommand("re_set_angle_lerp "..interp) end end},
	{"sublabel","Position Interpolation"},
	{"multibutton",{"none","linear","spline"},function(interp) if interp == "spline" then consolePerformCommand("re_set_position_lerp hermite") else consolePerformCommand("re_set_position_lerp "..interp) end end},
	{"toggle","Camera Node List",function() local re_show_camera_node_list = (consoleGetVariable("re_show_camera_node_list") == 0) consolePerformCommand(string.format("re_show_camera_node_list %i",re_show_camera_node_list and 1 or 0)) end,function() return (consoleGetVariable("re_show_camera_node_list") >= 1) end},
	{"divider"},
	{"label","Replay Controls"},
	{"button","Jump to next marker",getConsoleCommandFunc("ui_demoflex_cmd_exec re_next_marker")},
	{"button","Jump to prev marker",getConsoleCommandFunc("ui_demoflex_cmd_exec re_prev_marker")},
	{"divider"},
	{"button","Save Changes",function() if not Demoflex_Cmd.drawMessageBox then Demoflex_Cmd:pushMessageBox(Demoflex_Sidebar.mbSave,(replayNameBeforeRename == replayName) and replayNameAfterRename or replayName) Demoflex_Sidebar.showSave = true end end,function() return Demoflex_Sidebar.showSave end},
	{"button","Reset",function() if not Demoflex_Cmd.drawMessageBox then Demoflex_Cmd:pushMessageBox(Demoflex_Sidebar.mbReset) Demoflex_Sidebar.showReset = true end end,function() return Demoflex_Sidebar.showReset end},
	{"toggle","Advanced"},
}

Demoflex_Sidebar.tabPosition = 0.5
Demoflex_Sidebar.totalHoldTime = 1

function Demoflex_Sidebar:drawTab()
	local sidebarHoldTime = self.totalHoldTime
	local tabLabel = "DEMOFLEX CONTROLS"
	local tabHeight = fontSizeTab
	nvgSave()
	local tabMouseRegion = mouseRegion(-tabHeight, 0, tabHeight, viewport.height, 2)
	local mouseX = tabMouseRegion.mousex
	local mouseY = tabMouseRegion.mousey
	local tabPosition = self.tabPosition
	local tabClickY = self.tabClickY

	if viewport.height % 2 == 1 then nvgTranslate(0,-0.5) end
	nvgTranslate(0,viewport.height*tabPosition)
	if tabMouseRegion.leftDown then
		tabClickY = mouseY
		if self.holdTime ~= -1 then self.holdTime = sidebarHoldTime end
	elseif tabClickY and tabMouseRegion.leftHeld then
		nvgTranslate(0,mouseY-tabClickY)
		if self.holdTime ~= -1 then self.holdTime = sidebarHoldTime end
	elseif tabClickY and tabMouseRegion.leftUp then
		nvgTranslate(0,mouseY-tabClickY)
		tabPosition = tabPosition+(mouseY-tabClickY)/viewport.height
		if mouseY == tabClickY then if self.holdTime ~= -1 then self.holdTime = -1 else self.holdTime = sidebarHoldTime end end
	end
	nvgFontFace(fontFaceLogo)
	nvgFontSize(tabHeight)
	nvgFontBlur(0)
	nvgTextAlign(NVG_ALIGN_LEFT,NVG_ALIGN_BOTTOM)
	local textWidth = nvgTextWidth(tabLabel)
	local tabWidth = textWidth * 4/3
	nvgRotate(-math.pi/2)
	nvgBeginPath()
	nvgMoveTo(-(tabWidth-(tabHeight*3/4))/2,-tabHeight)
	nvgLineTo((tabWidth-(tabHeight*3/4))/2,-tabHeight)
	nvgLineTo(tabWidth/2,0)
	nvgLineTo(-(tabWidth)/2,0)
	if self.holdTime == -1 then nvgFillColor(lightGreyLowAlpha) else nvgFillColor(blue) end
	nvgFill()

	if tabMouseRegion.hover then
		local tabMid = viewport.height*tabPosition
		if mouseX > 0 or mouseX < -tabHeight or mouseY > tabMid + (tabWidth + mouseX*3/4)/2 or mouseY < tabMid - (tabWidth + mouseX*3/4)/2 then tabMouseRegion.hover = false end
	end

	nvgFillColor(white)
	nvgText(-math.floor(0.5+textWidth/2),0,tabLabel)

	nvgRestore()
	self.tabMouseRegion = tabMouseRegion
	self.tabPosition = tabPosition
	self.tabClickY = tabClickY
end

function Demoflex_Sidebar:draw()
	if not initializedAll or not enabled or hidden or replayName == "" or replayName == "menu" then return end
	if re_edit_toggle then
		local controlId = 1
		local lastInteract = self.interact
		self.interact = false
		nvgSave()
		nvgTranslate(viewport.width/2,-viewport.height/2)
		local lastMouseRegion = self.mouseRegion
		local lastTabMouseRegion = self.tabMouseRegion
		local sidebarSlideTime = 0.35
		local sidebarHoldTime = self.totalHoldTime
		local slide = self.slideAmount or 0
		local hold = self.holdTime or 0
		if lastInteract or (lastMouseRegion and lastMouseRegion.hover) or (lastTabMouseRegion and lastTabMouseRegion.hover) then
			if slide < 1 then
				slide = math.min(1,slide + deltaTimeRaw/sidebarSlideTime)
				self.slideAmount = slide
			end
			if hold ~= -1 and hold < sidebarHoldTime then
				hold = sidebarHoldTime
				self.holdTime = hold
			end
		else
			if hold > 0 then
				hold = math.max(0,hold - deltaTimeRaw)
				self.holdTime = hold
				if slide < 1 then
					slide = math.min(1,slide + deltaTimeRaw/sidebarSlideTime)
					self.slideAmount = slide
				end
			elseif hold == -1 then
				if slide < 1 then
					slide = math.min(1,slide + deltaTimeRaw/sidebarSlideTime)
					self.slideAmount = slide
				end
			elseif slide > 0 then
				slide = math.max(0,slide - deltaTimeRaw/sidebarSlideTime)
				self.slideAmount = slide
			end
		end
		nvgTranslate(-sidebarWidth*scaleTransition(slide),0)
		self.mouseRegion = mouseRegion(0,0,sidebarWidth,viewport.height,1)
		self:drawTab()
		controlId = controlId + 1
		nvgBeginPath()
		nvgRect(0,0,sidebarWidth,viewport.height)
		nvgStrokeColor(blue)
		nvgStrokeWidth(2)
		nvgStroke()
		nvgFillColor(black)
		nvgFill()
		nvgFontBlur(0)
		nvgTranslate(0,viewport.height-bottomMargin)
		nvgTextAlign(NVG_ALIGN_LEFT,NVG_ALIGN_BOTTOM)
		for i=#sidebarControls,1,-1 do
			local control = sidebarControls[i]
			local controlType = control[1]
			if controlType == "label" then
				nvgFontFace(fontFaceBold)
				nvgFontSize(fontSizeL)
				nvgFillColor(veryLightGrey)
				nvgText(horizontalMargin,(0),control[2])
				nvgTranslate(0,-controlSizeL)
			elseif controlType == "sublabel" then
				nvgFontFace(fontFaceRegular)
				nvgFontSize(fontSizeM)
				nvgFillColor(veryLightGrey)
				nvgText(horizontalMargin,0,control[2])
				nvgTranslate(0,-controlSizeM)
			elseif controlType == "divider" then
				nvgTranslate(0,-sizeDivider*2+1)
				nvgBeginPath()
				nvgRect(0,sizeDivider-2,sidebarWidth,5)
				nvgFillColor(midGrey)
				nvgFill()
				nvgBeginPath()
				nvgRect(0,sizeDivider-1,sidebarWidth,3)
				nvgFillColor(blue)
				nvgFill()
			elseif controlType == "button" then
				nvgTranslate(0,-sizeL)
				nvgFontFace(fontFaceRegular)
				local click,interact = button(horizontalMargin,(sizeL-controlSizeL)/2,sidebarWidth-horizontalMargin*2,controlSizeL,controlId,fontSizeM,control[2],control[3],control[4] and control[4](),Demoflex_Cmd.drawMessageBox)
				if click then control[3]() end
				if interact and not self.interact then self.interact = true end
				controlId = controlId + 1
			elseif controlType == "toggle" then
				nvgTranslate(0,-sizeL)
				local click,interact = button(horizontalMargin,(sizeL-controlSizeL)/2,sidebarWidth-horizontalMargin*2,controlSizeL,controlId,fontSizeM,control[2],control[3],control[4] and control[4](),Demoflex_Cmd.drawMessageBox)
				if click then control[3]() end
				if interact and not self.interact then self.interact = true end
				controlId = controlId + 1
			elseif controlType == "multibutton" then
				nvgTranslate(0,-sizeS)
				local spacing = 5
				local radioSet = control[2]
				local numButtons = #radioSet
				local buttonWidth = (sidebarWidth-horizontalMargin*2-((numButtons-1)*spacing))/numButtons
				nvgFontFace(fontFaceRegular)
				nvgSave()
				nvgTranslate(horizontalMargin,math.floor(0.5+(sizeS-controlSizeS)/3))
				for j=1,numButtons do
					local label = radioSet[j]
					local click,interact = button(math.floor(0.5+(j-1)*(buttonWidth+spacing)),0,math.floor(0.5+buttonWidth),controlSizeS,controlId,fontSizeS,label,true,false,Demoflex_Cmd.drawMessageBox)
					if click then control[3](label) end
					if interact and not self.interact then self.interact = true end
					controlId = controlId + 1
				end
				nvgRestore()
			end
		end
		nvgRestore()
	end
end
